package Thruk::Utils::CLI::Command;

=head1 NAME

Thruk::Utils::CLI::Command - Command CLI module

=head1 DESCRIPTION

The command command displays the expanded command for a given host or service

=head1 SYNOPSIS

  Usage: thruk [globaloptions] command [-n] <hostname> [<service_description>]

=head1 OPTIONS

=over 4

=item B<help>

    print help and exit

=item B<-n>

    print resulting command only

=back

=cut

use warnings;
use strict;
use Getopt::Long ();

use Thruk::Utils::CLI ();
use Thruk::Utils::Log qw/:all/;

##############################################

=head1 METHODS

=head2 cmd

    cmd([ $options ])

=cut
sub cmd {
    my($c, $action, $commandoptions, undef, undef, $global_options) = @_;

    if(!$c->check_user_roles('authorized_for_admin')) {
        return("ERROR - authorized_for_admin role required", 1);
    }

    $c->stats->profile(begin => "_cmd_command($action)");

    # parse options
    my $opt = {};
    Getopt::Long::Configure('no_ignore_case');
    Getopt::Long::Configure('bundling');
    Getopt::Long::Configure('pass_through');
    Getopt::Long::GetOptionsFromArray($commandoptions,
       "n"       => \$opt->{'naked_command'},
    ) or do {
        return(Thruk::Utils::CLI::get_submodule_help(__PACKAGE__));
    };

    my $hostname    = shift @{$commandoptions};
    my $description = shift @{$commandoptions};

    return(Thruk::Utils::CLI::get_submodule_help(__PACKAGE__)) unless $hostname;

    my $backend = $global_options->{'backends'}->[0] || '';
    my($host, $service);

    my $hosts = $c->db->get_hosts( filter => [ { 'name' => $hostname } ] );
    $host = $hosts->[0];
    # we have more and backend param is used
    if( scalar @{$hosts} == 1 and defined $backend ) {
        for my $h ( @{$hosts} ) {
            if( $h->{'peer_key'} eq $backend ) {
                $host = $h;
                last;
            }
        }
    }
    if(!$host) {
        return("no such host '".$hostname."'\n", 1);
    }

    if($description) {
        my $services = $c->db->get_services( filter => [{ 'host_name' => $hostname }, { 'description' => $description }, ] );
        $service = $services->[0];
        # we have more and backend param is used
        if( scalar @{$services} == 1 and defined $services ) {
            for my $s ( @{$services} ) {
                if( $s->{'peer_key'} eq $backend ) {
                    $service = $s;
                    last;
                }
            }
        }
        if(!$service) {
            return("no such service '".$description."' on host '".$hostname."'\n", 1);
        }
    }

    my $command = $c->db->expand_command('host' => $host, 'service' => $service, 'source' => $c->config->{'show_full_commandline_source'}, 'obfuscate' => 0 );

    if($opt->{'naked_command'}) {
        _warn($command->{'note'}) if $command->{'note'};
        return($command->{'line_expanded'}."\n", 0);
    }

    my $msg;
    $msg .= 'Note:             '.$command->{'note'}."\n" if $command->{'note'};
    $msg .= 'Check Command:    '.$command->{'line'}."\n";
    $msg .= 'Expanded Command: '.$command->{'line_expanded'}."\n";

    $c->stats->profile(end => "_cmd_command($action)");
    return($msg, 0);
}

##############################################

=head1 EXAMPLES

Display expanded command for test service on host localhost

  %> thruk command localhost test


Note: This command is for retrieving expanded check command lines only.
Submitting external commands is possible via the 'url' command or the rest api. See 'thruk url help' for some examples.

=cut

##############################################

1;
